<?php
/*--------------------------------------------------------------------
 OnGetSellingUnitPriceEventListener.php 2020-2-28
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Shop\SellingUnit\Database\Price\PropertyInformation\Listener;

use Gambio\Shop\Product\ValueObjects\ProductId;
use Gambio\Shop\ProductModifiers\Modifiers\Collections\ModifierIdentifierCollectionInterface;
use Gambio\Shop\ProductModifiers\Modifiers\ValueObjects\ModifierIdentifierInterface;
use Gambio\Shop\SellingUnit\Database\Price\PropertyInformation\Exceptions\ProductDoesNotHaveAnyCombinationsException;
use Gambio\Shop\SellingUnit\Database\Price\PropertyInformation\Service\ReadService;
use Gambio\Shop\SellingUnit\Database\Price\PropertyInformation\Service\ReadServiceInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSellingUnitPriceEventInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PriceFormatted;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PricePlain;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PriceStatus;
use PropertiesCombisAjaxHandler;
use PropertiesControlInterface;
use stdClass;

/**
 * Class OnGetSellingUnitPriceEventListener
 * @package Gambio\Shop\SellingUnit\Database\Price\PropertyInformation\Listener
 */
class OnGetSellingUnitPriceEventListener
{
    /**
     * @var ReadServiceInterface
     */
    protected $readService;
    
    /**
     * @var PropertiesControlInterface
     */
    protected $propertiesControl;
    
    /**
     * @var PropertiesCombisAjaxHandler
     */
    protected $combisAjaxHandler;
    
    
    /**
     * OnGetSellingUnitPriceEventListener constructor.
     *
     * @param ReadServiceInterface        $readService
     * @param PropertiesControlInterface  $propertiesControl
     * @param PropertiesCombisAjaxHandler $combisAjaxHandler
     */
    public function __construct(
        ReadServiceInterface $readService,
        PropertiesControlInterface $propertiesControl,
        PropertiesCombisAjaxHandler $combisAjaxHandler
    ) {
        $this->readService       = $readService;
        $this->propertiesControl = $propertiesControl;
        $this->combisAjaxHandler = $combisAjaxHandler;
    }
    
    
    /**
     * @param OnGetSellingUnitPriceEventInterface $event
     */
    public function __invoke(OnGetSellingUnitPriceEventInterface $event)
    {
        $product    = $event->product();
        $productId  = $event->productId();
        $languageId = $event->languageId();
        $xtcPrice   = $event->xtcPrice();
        $modifiers  = $event->modifiers();
        $quantity   = $event->quantity()->value();
        
        try {
            $combiIds = $this->readService->availableCombisIds($productId);
        } catch (ProductDoesNotHaveAnyCombinationsException $exception) {
            
            unset($exception);
        }
    
        $cheapestCombi = $this->propertiesControl->get_cheapest_combi($productId->value(), $languageId->value());
        $xtcPrice->setShowFromAttributes(true);
    
        if (!empty($cheapestCombi) && $cheapestCombi['combi_price'] !== 0) {
            
            [
                'formated' => $priceFormatted,
                'plain'    => $pricePlain
            ] = $xtcPrice->xtcGetPrice($productId->value(),
                                       true,
                                       $quantity,
                                       $product->getTaxClassId(),
                                       $product->getPrice(),
                                       1,
                                       0,
                                       true,
                                       true,
                                       $cheapestCombi['products_properties_combis_id']);
        }
        
        if (isset($combiIds) && $combiIds !== null) {
            
            $combiId      = $combiIds[0]->combisId();
            $currentCombi = $this->propertiesControl->get_combis_full_struct($combiId, $languageId->value());
            
            [
                'formated' => $priceFormatted,
                'plain'    => $pricePlain
            ] = $xtcPrice->xtcGetPrice($productId->value(),
                                       true,
                                       $quantity,
                                       $product->getTaxClassId(),
                                       $product->getPrice(),
                                       1,
                                       0,
                                       true,
                                       true,
                                       $currentCombi['products_properties_combis_id']);
        }
        
        if (count($modifiers)) {
            
            $propertyValuesIds = $this->propertiesValuesIds($modifiers);
            
            if ($this->allPropertyValueIdsAreNumeric($propertyValuesIds)) {
                
                $selectionTemplate = $this->getPropertiesSelectionTemplate($quantity, $propertyValuesIds, $productId);
    
                if ($selectionTemplate instanceof stdClass && $selectionTemplate->status !== 'no_combi_selected') {
    
                    [
                        'price'       => $priceFormatted,
                        'plain_price' => $pricePlain
                    ] = (array)$selectionTemplate;
                }
            }
        }
        
        if (isset($priceFormatted, $pricePlain)) {
            
            $event->builder()
                ->withPricePlain($this->pricePlain($pricePlain))
                ->withPriceFormatted($this->priceFormatted($priceFormatted))
                ->withStatus($this->priceStatus($event->product()->priceStatus()));
        }
    }
    
    
    /**
     * @param array $propertyValuesIds
     *
     * @return bool
     */
    protected function allPropertyValueIdsAreNumeric(array $propertyValuesIds): bool
    {
        if (count($propertyValuesIds) === 0) {
            
            return false;
        }
        
        $isNumericResultArray = array_map(static function (int $propertyValueId) {
            return is_numeric($propertyValueId);
        },
            $propertyValuesIds);
        
        return in_array(false, $isNumericResultArray, true) === false;
    }
    
    
    /**
     * @param ModifierIdentifierCollectionInterface $modifiers
     *
     * @return array
     */
    protected function propertiesValuesIds(ModifierIdentifierCollectionInterface $modifiers): array
    {
        $propertyValuesIds = [];
    
        foreach ($modifiers as $modifier) {
            
            if ($modifier->type() === 'property') {
            
                $propertyValuesIds[] = $modifier->value();
            }
        }
        
        return $propertyValuesIds;
    }
    
    
    /**
     * @param float     $quantity
     * @param array     $propertiesValuesIds
     * @param ProductId $productId
     *
     * @return mixed
     */
    protected function getPropertiesSelectionTemplate(float $quantity, array $propertiesValuesIds, ProductId $productId)
    {
        $this->combisAjaxHandler->set_data('GET', ['action' => 'get_selection_template']);
        $this->combisAjaxHandler->set_data('POST',
                                           [
                                               'properties_values_ids' => $propertiesValuesIds,
                                               'quantity'              => $quantity,
                                               'products_id'           => $productId->value()
                                           ]);
        $this->combisAjaxHandler->proceed();
        header('Content-type: text/html');
        
        return json_decode($this->combisAjaxHandler->get_response());
    }
    
    /**
     * @param float $pricePlain
     *
     * @return PricePlain
     */
    protected function pricePlain(float $pricePlain): PricePlain
    {
        return new PricePlain($pricePlain);
    }
    
    
    /**
     * @param string $priceFormatted
     *
     * @return PriceFormatted
     */
    protected function priceFormatted(string $priceFormatted): PriceFormatted
    {
        return new PriceFormatted($priceFormatted);
    }

    /**
     * @param int $priceStatus
     * @return PriceStatus
     */
    protected function priceStatus(int $priceStatus) : PriceStatus
    {
        return new PriceStatus($priceStatus);

    }
}